<?php

/**
 * This file is part of Swoft.
 *
 * @link https://swoft.org
 * @document https://doc.swoft.org
 * @contact group@swoft.org
 * @license https://github.com/swoft-cloud/swoft/blob/master/LICENSE
 */

namespace Hdll\Services\Reservation\Lib;

use Swoft\Core\ResultInterface;

/**
 * The interface of Reservation service
 *
 * @method ResultInterface deferGetReservById(int $storeId, int $id)
 * @method ResultInterface deferIsReserved(int $storeId, int $cmanId, int $reservTime)
 * @method ResultInterface deferGetReservListByOrderId(int $storeId, int $orderId, $state = null)
 * @method ResultInterface deferGetReservListByOrderIds(int $storeId, array $orderIds, $state = null)
 * @method ResultInterface deferModifyReservById(int $storeId, int $id, array $data)
 * @method ResultInterface defermarkReservFinished(int $storeId, int $orderId)
 * @method ResultInterface deferAddReservation(int $storeId, int $orderId, int $buyerId, int $cmanId, int $reservTime, string $memo, int $type)
 * @method ResultInterface deferCancelReservByOrderId(int $storeId, int $orderId, string $memo)
 */
interface ReservationInterface
{
    /**
     * 获取某个预约的信息
     *
     * @param integer $storeId, 
     * @param integer $id
     * @return array
     */
    public function getReservById(int $storeId, int $id);

    /**
     * 检查指定的时间是否已被预约过
     * 
     * @param integer $storeId    店铺id
     * @param integer $cmanId     手艺人id，查询某个手艺人下的预约情况，$cmanId == 0表示不指定手艺人，
     * @param integer $reservTime 要检查的时间点，时间戳格式
     * @Number(name="cmanId")
     * @Number(name="reservTime")
     * @return boolen true:已预约，false:未预约
     */
    public function isReserved(int $storeId, int $cmanId, int $reservTime);

    /**
     * 根据订单id，获取商家某个订单包含的所有预约
     *
     * @param integer $storeId
     * @param integer $orderId
     * @param integer $state
     * @return array
     */
    public function getReservListByOrderId(int $storeId, int $orderId, $state = null);

    /**
     * 根据订单id列表，获取对应的预约列表
     *
     * @param integer $storeId
     * @param integer $orderIds
     * @param integer $state
     * @Number(name="state")
     * @return array
     */
    public function getReservListByOrderIds(int $storeId, array $orderIds, $state = null);

    /**
     * 修改某个预约信息
     *
     * @param integer $storeId, 
     * @param integer $id
     * @param array $data
     * $data 字段说明：
     * ['state'] => 预约状态，默认0，如果正常结束则为1
     * ['reservTime'] => 预约时间，整形，时间戳格式
     * ['memo'] => 预约备注
     * @return int 返回成功修改的条数
     */
    public function modifyReservById(int $storeId, int $id, array $data);

    /**
     * 根据orderId标记订单下所有预约的状态为“已完成”
     *
     * @param integer $storeId
     * @param integer $orderId
     * @Number(name="orderId")
     * @return bool 返回操作结果，true操作成功
     */
    public function markReservFinished(int $storeId, int $orderId);

    /**
     * 添加预约
     *
     * @param integer $storeId
     * @param integer $orderId
     * @param integer $buyerId
     * @param integer $cmanId      手艺人id
     * @param integer $reservTime  预约的时间
     * @param string $memo         预约备注
     * @param integer $type        预约类型，传值类型参加 ../Enum/ReservEnum.php
     * @return integer 添加成功则返回记录id
     */
    public function addReservation(int $storeId, int $orderId, int $buyerId, int $cmanId, int $reservTime, string $memo, int $type);

    /**
     * 根据订单id取消该订单下的所有预约
     *
     * @param integer $storeId
     * @param integer $orderId
     * @param string $memo
     * @return boolean 返回取消结果
     */
    public function cancelReservByOrderId(int $storeId, int $orderId, string $memo = '');

}