<?php
/**
 * This file is part of Swoft.
 *
 * @link https://swoft.org
 * @document https://doc.swoft.org
 * @contact group@swoft.org
 * @license https://github.com/swoft-cloud/swoft/blob/master/LICENSE
 */

namespace Hdll\Services\Store\Lib;

use Swoft\Core\ResultInterface;

/**
 * The interface of store service
 *
 * @method ResultInterface deferIsCertified(int $storeId)
 * @method ResultInterface deferIsNormal(int $storeId)
 * @method ResultInterface deferGetStoreByStoreId(int $storeId, array $fields=[])
 * @method ResultInterface deferGetStoreByStoreName(string $storeName, array $fields=[])
 * @method ResultInterface deferGetStoreBySellerId(int $sellerId, array $fields=[])
 * @method ResultInterface deferGetListBySellerIds(array $sellerIds, array $fields = [])
 * @method ResultInterface deferGetListByStoreIds(array $storeIds, array $fields = [])
 * @method ResultInterface deferGetListByLastMonth()
 * @method ResultInterface deferGetStoreByMobile(int $mobile, array $fields = [])
 * @method ResultInterface deferUpdateStore(int $storeId, array $data)
 * @method ResultInterface deferAddVipTime(int $storeId, int $days)
 * @method ResultInterface deferDeleteStore(int $storeId)
 * @method ResultInterface deferGetStoreCount(int $startTime, int $endTime, $edition=null, $oemId=0)
 * @method ResultInterface deferGetWxacodeUrl(int $storeId, int $type, string $page, string $scene)
 * @method ResultInterface deferGetByOemIdAndMobile(int $oemId, $mobile, array $fields = [])
 */
interface StoreInterface
{
    /**
     * 查询店铺是否已实名认证（企业认证）
     *
     * @param integer $storeId
     * @return bool true:认证成功  false:未认证
     */
    public function isCertified(int $storeId);

    /**
     * 查询店铺是否正常
     *
     * @param integer $storeId
     * @return bool true:正常，false:冻结
     */
    public function isNormal(int $storeId);

    /**
     * 根据店铺ID获取店铺信息
     *
     * @param integer $storeId
     * @return array 空数组表示没有查询到记录
     */
    public function getStoreByStoreId(int $storeId, array $fields=[]);

    /**
     * 根据店铺名称获取店铺信息
     *
     * @param string $storeName
     * @return array
     */
    public function getStoreByStoreName(string $storeName, array $fields=[]);

    /**
     * 根据卖家ID获取店铺信息
     *
     * @param integer $sellerId
     * @return array
     */
    public function getStoreBySellerId(int $sellerId, array $fields=[]);

    /**
     * 根据卖家ID列表获取店铺信息列表
     *
     * @param array $sellerIds
     * @return array
     * sellerName   卖家姓名
     * mobile       注册手机号
     * edition      是否VIP  2 表示是VIP
     */
    public function getListBySellerIds(array $sellerIds, array $fields = []);

    /**
     * 根据店铺ID列表获取店铺信息列表
     *
     * @param array $storeIds
     * @param array $fields
     * @return array
     */
    public function getListByStoreIds(array $storeIds, array $fields = []);

    /**
     * 获取最近一个月登录的店铺列表
     *
     * @return array
     * array(14) {
     *   [0]=>
     *   array(4) {
     *       ["id"]=>
     *       int(11)
     *       ["sellerId"]=>
     *       int(8)
     *       ["storeName"]=>
     *       string(12) "平凡小店"
     *       ["unionId"]=>
     *       string(28) "ohQUyuBp5CWZjyPJEsG4znbFQIZI"
     *   }
     * }
     */
    public function getListByLastMonth();

    /**
     * 根据注册手机号获取店铺信息
     *
     * @param integer $mobile
     * @param array $fields 要查询的字段列表 ：['id','storeName','addTime']
     * @return array
     */
    public function getStoreByMobile(int $mobile, array $fields = []);
    
    /**
     * 根据oemId和手机号获取店铺信息
     *
     * @param integer $oemId
     * @param integer $mobile
     * @return array
     */
    public function getByOemIdAndMobile(int $oemId, $mobile, array $fields = []);

    /**
     * 根据店铺ID修改店铺信息
     *
     * @param integer $storeId
     * @param array $data
     * 参数 $data 字段说明：
     * ['storeName'] => 店铺名称
     * ['mobile'] => 店铺绑定的手机号
     * ['wechat'] => 店铺绑定的微信号
     * ['edition'] => 订购版本，1普通版，2VIP版
     * ['storeAddr'] => 店铺地址
     * ['lntLat'] => 店铺地址，坐标信息
     * ['logoUrl'] => 店铺logo图片url
     * ['bannerUrl'] => 店铺banner 图片Url
     * ['templateId'] => 当前使用的个性模板id
     * ['storeIntro'] => 店铺介绍
     * ['storeHours'] => 店铺营业时间，示例："09:00-22:00" 表示从9点到22点
     * ['vipStartTime'] => VIP订购周期，开始时间，整数类型
     * ['vipEndTime'] => VIP订购周期，结束时间，整数类型
     * @return int  成功更新的条数
     */
    public function updateStore(int $storeId, array $data);

    /**
     * 根据店铺ID增加店铺VIP时间
     *
     * @param integer $storeId
     * @param array $days 需要增加的天数
     * @return int  成功更新的条数
     */
    public function addVipTime(int $storeId, int $days);

    /**
     * 根据店铺id删除一个店铺记录
     *
     * @param integer $storeId
     * @return int 成功返回删除条数
     */
    public function deleteStore(int $storeId);

    /**
     * 生成店铺小程序码
     *
     * @param integer $type 指明哪小程序，1前台小程序，2后台小程序
     * @param string $page  要跳转的小程序页面，如：pages/index/index
     * @param string $scene 通过小程序码传递的参数
     * @return string 成功返回小程序码的图片资源名（腾讯云存储的资源名，如：store_wxacode/9a414ff3b4c478a36c2ba7df563ee88b.png）
     */
    public function getWxacodeUrl(int $storeId, int $type, string $page, string $scene);

    /**
     * 获取指定时间段内的有效店铺数量
     *
     * @param integer $startTime
     * @param integer $endTime
     * @param integer $edition
     * @return integer
     */
    public function getStoreCount(int $startTime, int $endTime, $edition=null, $oemId=0);

}