<?php declare(strict_types=1);
namespace Hdll\Services\V2\Common\ClsLogger;

use DateTime;
use Monolog\Handler\AbstractProcessingHandler;
use Swoft\Co;
use Swoft\Log\Helper\Log;
use Swoft\Log\Logger;
use Swoft\Log\Logger as SwoftLogger;
use Swoft\Stdlib\Helper\JsonHelper;
use function array_column;
use function implode;
use function in_array;

/**
 * Class FileHandler
 *
 * @since 2.0
 */
class LogHandler extends AbstractProcessingHandler
{
    //腾讯云日志主题
    protected $topicId;

    /**
     * Write log levels
     *
     * @var string
     */
    protected $levels = '';

    /**
     * @var array
     */
    protected $levelValues = [];

    /**
     * Will exec on construct
     */
    public function init(): void
    {
        if (is_array($this->levels)) {
            $this->levelValues = $this->levels;
            return;
        }

        // Levels like 'notice,error'
        if (is_string($this->levels)) {
            $levelNames        = explode(',', $this->levels);
            $this->levelValues = SwoftLogger::getLevelByNames($levelNames);
        }
    }

    /**
     * Write log by batch
     *
     * @param array $records
     *
     * @return void
     */
    public function handleBatch(array $records): void
    {
        $records = $this->recordFilter($records);
        if (!$records) {
            return;
        }

        $this->write($records);
    }

    /**
     * Write file
     *
     * @param array $records
     */
    protected function write(array $records): void
    {
        if (Log::getLogger()->isJson()) {
            $records = array_map([$this, 'formatJson'], $records);
        } else {
            $records = array_column($records, 'formatted');
        }
        $messageText = implode(" __||__ ", $records);

        if($this->topicId == ClsLog::TOPICID_ERROR) {
            if (Co::id() > 0) {
                sgo(function () use ($messageText) {
                    ClsLog::error($messageText);
                });
            } else {
                ClsLog::error($messageText);
            }
        } else if($this->topicId == ClsLog::TOPICID_NOTICE) {
            if (Co::id() > 0) {
                sgo(function () use ($messageText) {
                    ClsLog::notice($messageText);
                });
            } else {
                ClsLog::notice($messageText);
            }
        }
    }

    /**
     * Filter record
     *
     * @param array $records
     *
     * @return array
     */
    private function recordFilter(array $records): array
    {
        $messages = [];
        foreach ($records as $record) {
            if (!isset($record['level'])) {
                continue;
            }
            if (!$this->isHandling($record)) {
                continue;
            }

            $record = $this->processRecord($record);
            $search1 = strpos($record['messages'], '[http://testapi.2b3.cn:80/]');
            $search2 = strpos($record['messages'], '[http://api.2b3.cn:80/]');
            if($search1 !== false || $search2 !== false) {
                continue;
            }
            $record['formatted'] = $this->getFormatter()->format($record);

            $messages[] = $record;
        }
        return $messages;
    }

    /**
     * @param array $record
     *
     * @return string
     */
    public function formatJson(array $record): string
    {
        unset($record['formatted'], $record['extra']);
        if ($record['level'] === Logger::NOTICE) {
            unset($record['context']);
        }

        if ($record['datetime'] instanceof DateTime) {
            $record['datetime'] = $record['datetime']->format('Y-m-d H:i:s');
        }

        return JsonHelper::encode($record, JSON_UNESCAPED_UNICODE);
    }

    /**
     * Whether to handler log
     *
     * @param array $record
     *
     * @return bool
     */
    public function isHandling(array $record): bool
    {
        if (empty($this->levelValues)) {
            return true;
        }

        return in_array($record['level'], $this->levelValues, true);
    }

}
