<?php
/**
 * This file is part of Swoft.
 *
 * @link https://swoft.org
 * @document https://doc.swoft.org
 * @contact group@swoft.org
 * @license https://github.com/swoft-cloud/swoft/blob/master/LICENSE
 */

namespace Hdll\Services\Order\Lib;

use Swoft\Core\ResultInterface;

/**
 * The interface of demo service
 *
 * @method ResultInterface deferCreateOrder(int $buyerId,int $storeId,int $consigneeId,int $itemId,int $selectedNum,int $orderType,int $activityPrice,string $goodsName,string $goodsImage,string $activityName,int $goodsId,int $goodsPrice,string $message='',int $skuId=0,int $subStoreId=0,int $couponId=0);
 * @method ResultInterface deferGetOrderDetailBySubId(int $storeId,int $subOrderId,$ifSelf=true,bool $ifExtend=false)
 * @method ResultInterface deferGetOrderDetailByPid(int $storeId,int $pId, $ifSelf=false, $ifExtend=false)
 * @method ResultInterface deferGetOrderDetailByMainSn(int $storeId,string  $orderSn, $ifSelf=false, $ifExtend=false)
 * @method ResultInterface deferUpdateMainOrderByPid(int $storeId,int $pId,array $mainData,array $subData=[])
 * @method ResultInterface deferUpdateInfoById(int $storeId,int $orderId, array  $updateInfo)
 * @method ResultInterface deferUpdateInfoByIds(int $storeId,array $orderIds, array  $updateInfo)
 * @method ResultInterface deferCollectionClose(int $storeId, int $orderId)
 * @method ResultInterface deferGetSubOrderListBySubIds(int $storeId,array $subIds)
 * @method ResultInterface deferUnconfirmedCount(int $storeId)
 */
interface OrderInterface
{
    /**
     * 创建订单
     *
     * @param integer $buyerId
     * @param integer $storeId
     * @param integer $consigneeId
     * @param integer $itemId
     * @param integer $selectedNum
     * @param integer $orderType
     * @param integer $activityPrice
     * @param string $goodsName
     * @param string $goodsImage
     * @param string $activityName
     * @param integer $goodsId
     * @param integer $goodsPrice
     * @param string $message
     * @param integer $skuId
     * @param integer $subStoreId
     * @param integer $couponId
     * @return mixed
     */
    public function createOrder(
        int     $buyerId,
        int     $storeId,
        int     $consigneeId,
        int     $itemId,
        int     $selectedNum,
        int     $orderType,
        int     $activityPrice,
        string  $goodsName,
        string  $goodsImage,
        string  $activityName,
        int     $goodsId,
        int     $goodsPrice,
        string  $message = '',
        int     $skuId=0,
        int $subStoreId=0,
        int $couponId=0
    );
    /**
    * 根据子订单ID查询订单详情
    *
    * @param integer $storeId 店铺ID
    * @param integer $subOrderId 子订单ID
    * @param boolean $ifSelf  子订单是否只有自己
    * @param boolean $ifExtend 是否涵带附加信息（如门店信息、上级信息等）
    * @return array
    */
    public function getOrderDetailBySubId(int $storeId,int $subOrderId,$ifSelf=true,bool $ifExtend=false);
    /**
    * 根据主订单ID查询订单详情
    *
    * @param integer $storeId 店铺ID
    * @param integer $pId 主订单ID
    * @param boolean $ifExtend 是否涵带附加信息（如门店信息、上级信息等）
    * @return array
    */
    public function getOrderDetailByPid(int $storeId,int $pId, $ifSelf=false, $ifExtend=false);
    
    /**
    * 通过订主单orderSn获取订单信息
    *
    * @param integer $storeId
    * @param string $orderSn
    * @param boolean $ifSelf 是否只有主订单信息
    * @param boolean $ifExtend 是否涵带附加信息（如门店信息、上级信息等）
    * @return array
    */
    public function getOrderDetailByMainSn(int $storeId,string  $orderSn, $ifSelf=false, $ifExtend=false);

    /**
     * 根据订单ID更新子订单信息
     *
     * @param integer $storeId
     * @param integer $orderId
     * @param array $updateInfo
     * @return mixed
     */
    public function updateInfoById(int $storeId,int $orderId, array  $updateInfo);

    /**
     * 批量更新
     *
     * @param int $storeId
     * @param array $orderIds
     * @param array $updateInfo
     * @return mixed
     */
    public function updateInfoByIds(int $storeId,array $orderIds, array  $updateInfo);

    /**
     * 根据主订单ID更新订单信息
     *
     * @param integer $storeId
     * @param integer $pId
     * @param array $mainData
     * @param array $subData
     * @return bool
     */
    public function updateMainOrderByPid(int $storeId,int $pId,array $mainData,array $subData=[]);

    /**
     * 退款成功
     *
     * @param int $storeId
     * @param array $orderInfo
     * @return mixed
     */
    public function refundSuccess(int $storeId, array $orderInfo);

    /**
     * 拼团关闭订单，含退款
     *
     * @param int $storeId
     * @param int $orderId
     * @return mixed
     */
    public function collectionClose(int $storeId, int $orderId);

    /**
     * 获取未入账统计
     * @param int $storeId
     * @return array
     */
    public function unconfirmedCount(int $storeId);

    /**
     * 批量查询子订单信息
     *
     * @param integer $storeId
     * @param array $subIds
     * @return void
     */
    public function getSubOrderListBySubIds(int $storeId,array $subIds);
}