<?php

/**
 * This file is part of Swoft.
 *
 * @link https://swoft.org
 * @document https://doc.swoft.org
 * @contact group@swoft.org
 * @license https://github.com/swoft-cloud/swoft/blob/master/LICENSE
 */

namespace Hdll\Services\Reservation\Lib;

use Swoft\Core\ResultInterface;

/**
 * The interface of Reservation service
 *
 * @method ResultInterface deferGetReservById(int $storeId, int $id)
 * @method ResultInterface deferGetReservListByOrderId(int $storeId, int $orderId, int $state = 0, int $type = 1)
 * @method ResultInterface deferModifyReservById(int $storeId, int $id, array $data)
 * @method ResultInterface deferAddReservation(int $storeId, int $orderId, int $buyerId, int $cmanId, int $reservTime, string $memo, int $type)
 * @method ResultInterface deferCancelReservById(int $storeId, int $id)
 * @method ResultInterface deferCancelReservByOrderId(int $storeId, int $orderId)
 */
interface ReservationInterface
{
    /**
     * 获取某个预约的信息
     *
     * @param integer $storeId, 
     * @param integer $id
     * @return array
     */
    public function getReservById(int $storeId, int $id);

    /**
     * 根据订单id，获取商家某个订单包含的所有预约
     *
     * @param integer $storeId
     * @param integer $orderId
     * @param integer $state
     * @param integer $type
     * @return array
     */
    public function getReservListByOrderId(int $storeId, int $orderId, int $state = 0, int $type = 1);

    /**
     * 修改某个预约信息
     *
     * @param integer $storeId, 
     * @param integer $id
     * @param array $data
     * $data 字段说明：
     * ['state'] => 预约状态，默认0，如果正常结束则为1
     * ['type'] => 预约类型，备用
     * ['reservTime'] => 预约时间，整形，时间戳格式
     * ['memo'] => 预约备注
     * @return int 返回成功修改的条数
     */
    public function modifyReservById(int $storeId, int $id, array $data);

    /**
     * 添加预约
     *
     * @param integer $storeId
     * @param integer $orderId
     * @param integer $buyerId
     * @param integer $cmanId      手艺人id
     * @param integer $reservTime  预约的时间
     * @param string $memo         预约备注
     * @param integer $type        预约类型，备用
     * @return integer 添加成功则返回记录id
     */
    public function addReservation(int $storeId, int $orderId, int $buyerId, int $cmanId, int $reservTime, string $memo, int $type = 1);

    /**
     * 取消某个预约
     *
     * @param integer $storeId
     * @param integer $id
     * @return boolean 返回取消结果
     */
    public function cancelReservById(int $storeId, int $id);

    /**
     * 根据订单id取消该订单下的所有预约
     *
     * @param integer $storeId
     * @param integer $orderId
     * @return boolean 返回取消结果
     */
    public function cancelReservByOrderId(int $storeId, int $orderId);

}